# -*- coding: utf-8 -*-
"""Exemples de Monte Carlo

**Exemples de simulation Monte-Carlo pour calculer une incertitude composée**

1. Estimation de la fréquence en connaissant la période $T=24.4$ µs. La précision de la mesure est estimée à $\Delta T = 0.1$ µs.
"""

import numpy as np
import matplotlib.pyplot as plt

# Entrez la période

T = 24.4e-6 # s

# Entrez la précision sur la période

DeltaT = 0.1e-6 # s

# Entrez la fonction de composition

def f(x):
    return 1/x

# Entrez le nombre de simulation que vous voulez effectuer 

N = 100000

# Calculs avec une distribution de probabilité uniforme
Periode=[]
Frequence=[]

for i in range(0,N):
    periode = np.random.uniform(T-DeltaT,T+DeltaT)
    Periode.append(periode)
    Frequence.append(f(periode))

plt.figure(1)
plt.hist(Periode,bins = 'rice')  # La commande 'rice' permet d'optimiser les intervalles d'affichage de l'histogramme
plt.title('Tirage aléatoire des périodes')
plt.xlabel('Périodes (s)')

plt.figure(2)
plt.hist(Frequence,bins = 'rice')
plt.title('Résultat du tirage aléatoire des fréquences après calcul')
plt.xlabel('Fréquences (Hz)')

# Calcul et affichage moyenne et écart type

moy = np.mean(Frequence)
std = np.std(Frequence,ddof=1)

print("Moyenne = {:.2f} Hz".format(moy))
print("Ecart type = {:.2f} Hz".format(std))

"""2. Simulation d'une incertitude de type somme : calcul de la distance entre deux points mesurés ($11.1$ cm et $24.8$ cm) avec une précision de $0.5$ cm."""

import numpy as np
import matplotlib.pyplot as plt

# Positions de des points

A = 11.1 # cm
B = 19.5 # cm

# Entrez les précisions

DeltaA = 0.5 # cm
DeltaB = 0.5 # cm

# Entrez la fonction de composition

def ecart(a,b):
    return b-a

# Entrez le nombre de simulation que vous voulez effectuer 

N = 100000

# Calculs avec une distribution de probabilité uniforme
Difference = []

for i in range(0,N):
    a = np.random.uniform(A-DeltaA,A+DeltaA)
    b = np.random.uniform(B-DeltaB,B+DeltaB)
    Difference.append(ecart(a,b))

plt.figure(1)
plt.hist(Difference,bins = 'rice')
plt.title('Résultat du tirage aléatoire de la différence après simulation')
plt.xlabel("b-a (cm)")

# Calcul et affichage moyenne et écart type

moy = np.mean(Difference)
std = np.std(Difference,ddof=1)

print("Moyenne = {:.2f} cm".format(moy))
print("Ecart type = {:.2f} cm".format(std))

######### Utilisation de la formule du programme #########

uA = DeltaA / np.sqrt(3)
uB = DeltaB / np.sqrt(3)

uDiff= np.sqrt(uA**2 + uB**2)


print("Incertitude de type somme avec la fomule = {:.2f} cm".format(uDiff))

"""3. Simulation d'une incertitude de type produit : calcul de la célérité du son avec une fréquence $f=40983$ Hz ($u(f)= 94$ Hz, estimée à partir de la mesure de période) et une longueur d'onde $λ = 0.840$ cm ($u(λ) = 0.041$ cm).

Attention, cette fois c'est directement l'incertitude-type qui est donnée (car elle est déduite des calculs précédents). Si on suppose que la distribution de probabilité est uniforme, il faut remonter à la précision à l'aide du facteur $\sqrt{3}$. Pour ne pas avoir à faire cette hypothèse, cf l'exemple 4.
"""

import numpy as np
import matplotlib.pyplot as plt

# Mesure

f = 40983 # Hz
lambd = 0.00840 # m

# Entrez les incertitudes types et précisions

uf = 94 # Hz
ulambd = 0.00041 # m

Deltaf=uf*np.sqrt(3)
Deltalambd=ulambd*np.sqrt(3)

# Entrez la fonction de composition

def produit(a,b):
    return a*b

# Entrez le nombre de simulation que vous voulez effectuer 

N = 100000

# Calculs avec une distribution de probabilité uniforme
Celerite = []

for i in range(0,N):
    periode = np.random.uniform(T-DeltaT,T+DeltaT)
    a = np.random.uniform(f-Deltaf,f+Deltaf)
    b = np.random.uniform(lambd-Deltalambd,lambd+Deltalambd)
    Celerite.append(produit(a,b))

plt.figure(1)
plt.hist(Difference,bins = 'rice')
plt.title('Résultat du tirage aléatoire du produit après simulation')
plt.xlabel("c (m/s)")

# Calcul et affichage moyenne et écart type

moy = np.mean(Celerite)
std = np.std(Celerite,ddof=1)

print("Moyenne = {:.2f} m/s".format(moy))
print("Ecart type = {:.2f} m/s".format(std))

######### Utilisation de la formule du programme #########

uProd= f*lambd*np.sqrt((uf/f)**2 + (ulambd/lambd)**2)


print("Incertitude de type produit avec la fomule = {:.2f} m/s".format(uProd))

"""4. Simulation d'une incertitude de type d'une mesure de célérité d'une onde. On mesure la fréquence $f$ et l'écart entre deux noeuds séparant 10 longueurs d'ondes. On reprend les valeurs et précisions des exemples 1 et 2."""

import numpy as np
import matplotlib.pyplot as plt

# Entrez la période

T = 24.4e-6 # s

# Entrez la précision sur la période

DeltaT = 0.1e-6 # s

# Positions de des points

A = 0.111 # m
B = 0.195 # m

# Entrez les précisions

DeltaA = 0.005 # m
DeltaB = 0.005 # m



# Entrez la fonction de composition

def celerite(T,a,b):
    return (b-a)/(10*T)

# Entrez le nombre de simulation que vous voulez effectuer 

N = 100000

# Calculs avec une distribution de probabilité uniforme
Celerite=[]

for i in range(0,N):
    x = np.random.uniform(T-DeltaT,T+DeltaT)
    a = np.random.uniform(A-DeltaA,A+DeltaA)
    b = np.random.uniform(B-DeltaB,B+DeltaB)
    Celerite.append(celerite(x,a,b))

plt.figure(1)
plt.hist(Difference,bins = 'rice')
plt.title('Résultat du tirage aléatoire du produit après simulation')
plt.xlabel("c (m/s)")

# Calcul et affichage moyenne et écart type

moy = np.mean(Celerite)
std = np.std(Celerite,ddof=1)

print("Moyenne = {:.2f} m/s".format(moy))
print("Ecart type = {:.2f} m/s".format(std))

"""On constate que les incertitudes-types des exemples 3 et 4 sont légerement différentes mais proches. Cela provient du fait qu'une hypothèse a été faite sur la distribution de probabilité des longueurs d'ondes et des fréquences dans l'exemple 3 alors qu'elle n'est pas nécessaire pour l'exemple 4.

5. Estimation d'une distance focale en connaissant la position de l'objet $\overline{OA} = - 15 cm$ (précision de la mesure $\Delta(\overline{OA})=0.5 cm$) et celle de l'image $\overline{OA'} = 30 cm$ (précision de la mesure $\Delta({OA'})=3 cm$).
"""

import numpy as np
import matplotlib.pyplot as plt

# Positions de l'objet et de l'image

OA = - 15 # cm
OAp = 30 # cm

# Entrez les précisions

DeltaOA = 1 # cm
DeltaOAp = 3 # cm

# Entrez la fonction de composition

def focale(objet,image):
    return 1/(1/image - 1/objet)

# Entrez le nombre de simulation que vous voulez effectuer 

N = 100000

# Calculs avec une distribution de probabilité uniforme
Focale = []

for i in range(0,N):
    objet = np.random.uniform(OA-DeltaOA,OA+DeltaOA)
    image = np.random.uniform(OAp-DeltaOAp,OAp+DeltaOAp)
    Focale.append(focale(objet,image))

plt.figure(1)
plt.hist(Focale,bins = 'rice')
plt.title('Résultat du tirage aléatoire des focales après simulation')
plt.xlabel("f' (cm)")


# Calcul et affichage moyenne et écart type

moy = np.mean(Focale)
std = np.std(Focale,ddof=1)

print("Moyenne = {:.2f} cm".format(moy))
print("Ecart type = {:.2f} cm".format(std))
